# AssistedKills release 4 by David Bowland
# ./addons/eventscripts/assistedkills/assistedkills.py

# >>> To configure this addon please see assistedkills.cfg <<<

"""
Fires a custom event when a player kills an opponent but did not cause the most damage to that opponent. Also fires when a player kills a blinded opponent but did not throw the flashbang. Requires only ES 2.0+
"""


import es
import langlib
import os.path
import playerlib
import time

import psyco
psyco.full()


info = es.AddonInfo()
info.name     = 'AssistedKills'
info.version  = '4'
info.url      = 'http://addons.eventscripts.com/addons/view/assistedkills'
info.basename = 'assistedkills'
info.author   = 'SuperDave'


###


class BlindedPlayersManager:
   players = {}

   def __init__(self):
      self.players = {}

   def addPlayer(self, userid):
      """Adds the player and player's flashduration to the dictionary"""
      self.players[userid] = es.getplayerprop(userid, 'CCSPlayer.m_flFlashDuration')

   def flashbangDetonate(self, userid):
      """Marks all players in the dictionary as blinded by the player whose flashbang detonated"""
      for loop_userid in self.players:
         getPlayer(loop_userid).playerBlind(userid, self.players[loop_userid])

      self.clearPlayers()

   def clearPlayers(self):
      """Clears the dictionary of blinded players"""
      self.players.clear()

blindedplayers = BlindedPlayersManager()


class AssistedKillManager:
   userid     = 0
   assists    = 0
   assisted   = 0
   blinded_by = 0
   end_blind  = 0
   he_count   = 0
   killed_by  = 0
   attackers  = {}

   def __init__(self, userid):
      self.userid    = userid
      self.attackers = {}

   def reset(self):
      """
      Resets player attributes
      Clears the dictionary of attackers
      """
      self._removeBlinded()

      self.he_count   = 0
      self.killed_by  = 0

      self.attackers.clear()


   def removeAttacker(self, attacker):
      """
      Removes attacker from the player dictionary
      Removes the delay if attacker blinded the player
      """
      if self.attackers.has_key(attacker):
         del self.attackers[attacker]

      if self.blinded_by == attacker:
         self._removeBlinded()


   def playerHurt(self, attacker, damage):
      """Stores the amount of damage each player takes and from whom"""
      attacker = attacker if attacker else self.userid
      self.attackers[attacker] = (self.attackers[attacker] if self.attackers.has_key(attacker) else 0) + damage


   def playerBlind(self, attacker, duration):
      """Marks the player as blinded by the attacker for the specified duration"""
      self.blinded_by = attacker
      self.end_blind  = time.time() + duration

   def getBlindAttacker(self):
      if not self.end_blind: return 0

      if time.time() < self.end_blind: return self.blinded_by

      self._removeBlinded()

      return 0

   def _removeBlinded(self):
      """Marks the player as no longer blind"""
      self.blinded_by = 0
      self.end_blind  = 0


   def throwGrenade(self):
      self.he_count += 1

   def heDetonate(self):
      if self.he_count: self.he_count -= 1

   def getHECount(self):
      return self.he_count

   def getKiller(self):
      return self.killed_by


   def playerDeath(self, attacker, weapon):
      """
      Compiles a list of assisting players
      Gives the killing player an assisted kill
      Gives each assisting player an assist kill
      Announces the death was assisted
      """
      self.killed_by = attacker
      if attacker:
         attacker_player = getPlayer(attacker)
      blind_attacker = self.getBlindAttacker()

      attacker_damage  = self.attackers[attacker] if self.attackers.has_key(attacker) else 0
      max_damage       = max(map(lambda x: self.attackers[x], self.attackers)) if self.attackers else 0
      assisters        = filter(lambda x: self.attackers[x] == max_damage, self.attackers) if max_damage >= attacker_damage + es.ServerVar('assistedkills_threshold') else []
      assist_flashbang = blind_attacker if blind_attacker and blind_attacker <> attacker and int(es.ServerVar('assistedkills_flashbang')) and blind_attacker not in assisters else 0

      assist_he = 0
      if noisy_state and attacker and weapon == 'hegrenade':
         if not attacker_player.getHECount() and es.getplayerprop(attacker, 'CBasePlayer.pl.deadflag'):
            assist_he = attacker_player.getKiller()

            if assist_he in assisters: assist_he = 0

      if not assisters and not assist_flashbang and not assist_he: return

      tokens       = {'assist':'', 'victim':es.getplayername(self.userid), 'damage':max_damage}
      assist_names = ''
      console      = int(es.ServerVar('assistedkills_announce_console'))
      center       = int(es.ServerVar('assistedkills_announce_center'))
      hud          = int(es.ServerVar('assistedkills_announce_hud'))
      chat         = int(es.ServerVar('assistedkills_announce_chat'))

      if attacker:
         attacker_player._assistedKill(self.userid)

      for userid in assisters:
         # Assisted kill damage
         self._fireAssistedDeath(attacker, userid, "DAMAGE", weapon, max_damage, attacker_damage)
         self._announceAssist('long text damage', userid, attacker, tokens, console, center, hud, chat)

      if blind_attacker:
         if assist_flashbang:
            # Assisted kill flashbang
            self._fireAssistedDeath(attacker, assist_flashbang, "FLASHBANG", weapon, 0, attacker_damage)
            self._announceAssist('long text flashbang', assist_flashbang, attacker, tokens, console, center, hud, chat)

            assisters.append(assist_flashbang)

         self._removeBlinded()

      if assist_he:
         # Assisted kill HE grenade
         self._fireAssistedDeath(attacker, assist_he, "HEGRENADE", weapon, 0, attacker_damage)
         self._announceAssist('long text hegrenade', assist_he, attacker, tokens, console, center, hud, chat)

         assisters.append(assist_he)

      tokens['assist'] = ', '.join(map(lambda x: es.getplayername(x), assisters))
      death_assist = func_lang_text('death assisted', tokens, playerlib.getPlayer(self.userid).get('lang'))
      if attacker:
         kill_assist = func_lang_text('kill assisted', tokens, playerlib.getPlayer(attacker).get('lang'))

      if center == 2:
         if attacker:
            es.centertell(attacker, self.__removeTags(kill_assist))
         es.centertell(self.userid, self.__removeTags(death_assist))

      if hud == 2:
         if attacker:
            ak_attacker.__hudHint(self.__removeTags(kill_assist))
         self.__hudHint(self.__removeTags(death_assist))

      if chat == 2:
         if attacker:
            es.tell(attacker, '#multi', kill_assist)
         es.tell(self.userid, '#multi', death_assist)


   def _fireAssistedDeath(self, attacker, assistid, assisttype, weapon, damage, attacker_damage):
      """Fires the player_assisteddeath event"""
      ak_assister = getPlayer(assistid)
      ak_assister._assistKill(self.userid)

      es.event('initialize', 'player_assisteddeath')
      es.event('setint',     'player_assisteddeath', 'userid',         self.userid)
      es.event('setint',     'player_assisteddeath', 'attacker',       attacker)
      es.event('setint',     'player_assisteddeath', 'assistid',       assistid)
      es.event('setstring',  'player_assisteddeath', 'assisttype',     assisttype)
      es.event('setstring',  'player_assisteddeath', 'weapon',         weapon)
      es.event('setint',     'player_assisteddeath', 'damage',         damage)
      es.event('setint',     'player_assisteddeath', 'attackerdamage', attacker_damage)
      es.event('setint',     'player_assisteddeath', 'assistcount',    ak_assister.assists)
      es.event('setint',     'player_assisteddeath', 'assistedcount',  getPlayer(attacker).assisted if attacker else 0)
      es.event('fire',       'player_assisteddeath')

   def _announceAssist(self, long_text, assistid, attacker, tokens, console, center, hud, chat):
      """Announces the assisted kill"""
      tokens['assist'] = es.getplayername(assistid)
      assist_message   = func_lang_text('assisted kill', tokens, playerlib.getPlayer(assistid).get('lang'))

      if console:
         if console == 2:
            for loop_userid in es.getUseridList():
               es.cexec(loop_userid, 'echo ' + self.__removeTags(func_lang_text(long_text, tokens, playerlib.getPlayer(loop_userid).get('lang'))))
         else:
            for loop_userid in filter(lambda x: bool(x), (self.userid, attacker, assistid)):
               es.cexec(loop_userid, 'echo ' + self.__removeTags(func_lang_text(long_text, tokens, playerlib.getPlayer(loop_userid).get('lang'))))

      if center:
         es.centertell(assistid, self.__removeTags(assist_message))

      if hud:
         getPlayer(assistid).__hudHint(self.__removeTags(assist_message))

      if chat:
         if chat == 3:
            for loop_userid in es.getUseridList():
               es.tell(loop_userid, '#multi', func_lang_text(long_text, tokens, playerlib.getPlayer(loop_userid).get('lang')))
         else:
            es.tell(assistid, '#multi', assist_message)

   def _assistKill(self, victim):
      """Adds an assist kill to the player's stats"""
      if es.getplayerteam(self.userid) <> es.getplayerteam(victim):
         self.assists += 1

         kills_to_points = int(es.ServerVar('assistedkills_reward_killstopoints'))
         reward_points   = int(es.ServerVar('assistedkills_reward_points'))
         if kills_to_points and reward_points:
            if not self.assists % kills_to_points:
               self.__applyScore(reward_points)

         reward_money = int(es.ServerVar('assistedkills_reward_money'))
         if reward_money:
            int_money = es.getplayerprop(self.userid, 'CCSPlayer.m_iAccount') + reward_money
            es.setplayerprop(self.userid, 'CCSPlayer.m_iAccount', int_money if int_money < 16000 else 16000)

   def _assistedKill(self, victim):
      """Adds an assisted kill to the player's stats"""
      if es.getplayerteam(self.userid) <> es.getplayerteam(victim):
         self.assisted += 1

         kills_to_points   = int(es.ServerVar('assistedkills_punishment_killstopoints'))
         punishment_points = int(es.ServerVar('assistedkills_punishment_points'))
         if kills_to_points and punishment_points:
            if not self.assisted % kills_to_points:
               self.__applyScore(punishment_points * -1)

         punishment_money = int(es.ServerVar('assistedkills_punishment_money'))
         if punishment_money:
            int_money = es.getplayerprop(self.userid, 'CCSPlayer.m_iAccount') - punishment_money
            es.setplayerprop(self.userid, 'CCSPlayer.m_iAccount', int_money if int_money > 0 else 0)


   def __applyScore(self, score):
      """
      Ensures there is one game_score entity
      Adds the points to the player's score
      """
      game_scores = es.createentitylist('game_score')

      if len(game_scores) > 1:
         for int_index in game_scores.keys()[1:]:
            es.server.cmd('es_xremove %s' % int_index)
      elif not len(game_scores):
         es.server.cmd('es_xentcreate %s game_score' % self.userid)

      es.server.queuecmd('es_xfire %s game_score addoutput \"spawnflags 1\"' % self.userid)
      es.server.queuecmd('es_xfire %s game_score addoutput \"points %s\"' % (self.userid, score))
      es.server.queuecmd('es_xfire %s game_score applyscore' % self.userid)

   def __hudHint(self, message):
      """Sends the player a HUD hint"""
      es.usermsg('create', 'assistedkills_hud', 'HintText')
      es.usermsg('write', 'short', 'assistedkills_hud', -1)
      es.usermsg('write', 'string', 'assistedkills_hud', message)
      es.usermsg('send', 'assistedkills_hud', self.userid)
      es.usermsg('delete', 'assistedkills_hud')

   def __removeTags(self, text):
      """Removes #lightgreen, #green and #default tags from the supplied string"""
      return text.replace('#lightgreen', '').replace('#green', '').replace('#default', '')


players = {}


def getPlayer(userid):
   """Returns an AssistedKillManager instance based on userid"""
   global players

   userid = int(userid)
   if not players.has_key(userid):
      players[userid] = AssistedKillManager(userid)

   return players[userid]


def removePlayer(userid):
   """Removes AssistedKillManager instance based on userid"""
   global players

   userid = int(userid)
   if players.has_key(userid):
      del players[userid]

   for loop_userid in players:
      players[loop_userid].removeAttacker(userid)


def resetPlayers():
   """
   Clears all AssistedKillManager instances
   """
   players.clear()


###


dict_events    = {'player_assisteddeath':{'userid':'short', 'attacker':'short', 'assistid':'short', 'assisttype':'string', 'weapon':'string', 'damage':'short', 'attackerdamage':'short', 'assistcount':'short', 'assistedcount':'short'}}
dict_options   = {'assistedkills_threshold':[15, 'Amount of damage a player must have more than the killing player to receive an assisted kill'], 'assistedkills_flashbang':[1, '0 = no flashbang assisted kills, 1 = blinding a killed player is considered assisting in the kill'], 'assistedkills_hegrenade':[1, '0 = no HE grenade assisted kills, 1 = killing a player who drops a grenade that kills another player is considered assisting in the latter kill'], 'assistedkills_reward_money':[150, 'Money a player receives for an assisted kill'], 'assistedkills_reward_killstopoints':[2, 'Number of assisted kills a player needs to receive points added to his or her score'], 'assistedkills_reward_points':[1, 'Number of points a player will receive for the requisite number of assisted kills'], 'assistedkills_punishment_money':[150, 'Money a player loses for a kill that was assisted'], 'assistedkills_punishment_killstopoints':[5, 'Number of kills that can be assisted until player loses points from his or her score'], 'assistedkills_punishment_points':[2, 'Number of points a player will lose for the requisite number of kills that are assisted'], 'assistedkills_announce_console':[2, '0 = no change, 1 = announce assisted kills to involved players on the console, 2 = announce assisted kills to all players on the console'], 'assistedkills_announce_center':[2, '0 = no change, 1 = announce assisted kills to the player who made the assisted kill in center text, 2 = announce assisted kills to players involved in the assisted kill in center text'], 'assistedkills_announce_hud':[0, '0 = no change, 1 = announce assisted kills to the player who made the assisted kill in a HUD message, 2 = announce assisted kills to players involved in the assisted kill in a HUD message'], 'assistedkills_announce_chat':[0,'0 = no change, 1 = announce assisted kills to the player who made the assisted kill in chat area, 2 = announce assisted kills to players involved in the assisted kill in chat area, 3 = announce assisted kills to all players in chat area']}
func_lang_text = lambda x, y, z: 'AssistedKills: assistedkills_languages.ini not found'
noisy_state    = False


def load():
   global func_lang_text
   global noisy_state

   for str_option in dict_options:
      es.ServerVar(str_option, dict_options[str_option][0], dict_options[str_option][1])
   if os.path.isfile(es.getAddonPath('assistedkills') + '/assistedkills.cfg'):
      es.server.cmd('es_xmexec ../addons/eventscripts/assistedkills/assistedkills.cfg')
   else:
      es.dbgmsg(0, 'AssistedKills: Unable to load assistedkills.cfg! Please ensure it is in the ./assistedkills/ directory.')

   str_lang_path = es.getAddonPath('assistedkills') + '/assistedkills_languages.ini'
   if os.path.isfile(str_lang_path):
      func_lang_text = langlib.Strings(str_lang_path)
   else:
      es.dbgmsg(0, 'AssistedKills: Unable to load assistedkills_languages.ini! Please ensure it is in the ./assistedkills/ directory.')

   load_events()

   if int(es.ServerVar('assistedkills_hegrenade')):
      es.doblock('corelib/noisy_on')
      noisy_state = True


def es_map_start(event_var):
   """
   Calls load_events
   Clears players so data is recreated each map
   """
   load_events()

   resetPlayers()


def round_start(event_var):
   blindedplayers.clearPlayers()


def player_spawn(event_var):
   getPlayer(event_var['userid']).reset()


def player_hurt(event_var):
   getPlayer(event_var['userid']).playerHurt(int(event_var['attacker']), int(event_var['dmg_health']))


def player_blind(event_var):
   blindedplayers.addPlayer(int(event_var['userid']))


def player_disconnect(event_var):
   removePlayer(event_var['userid'])


def player_death(event_var):
   getPlayer(event_var['userid']).playerDeath(int(event_var['attacker']), event_var['weapon'])


def weapon_fire(event_var):
   if event_var['weapon'] == 'hegrenade':
      getPlayer(event_var['userid']).throwGrenade()


def hegrenade_detonate(event_var):
   getPlayer(event_var['userid']).heDetonate()


def flashbang_detonate(event_var):
   blindedplayers.flashbangDetonate(int(event_var['userid']))


def unload():
   if noisy_state:
      es.doblock('corelib/noisy_off')

   for str_option in dict_options:
      es.ServerVar(str_option).set(0)


###


def load_events():
   """Loads the .res file for the player_assisteddeath event"""
   if os.path.isfile(es.getAddonPath('assistedkills') + '/assistedkills.res'):
      es.loadevents('declare', 'addons/eventscripts/assistedkills/assistedkills.res')

   else:
      if not os.path.isfile(es.getAddonPath('assistedkills') + 'es_assistedkills_events_db.txt'):
         es.keygroupcreate('assistedkills_events')
         for str_event in dict_events:
            es.keycreate('assistedkills_events', str_event)
            for str_event_var in dict_events[str_event]:
               es.keysetvalue('assistedkills_events', str_event, str_event_var, dict_events[str_event][str_event_var])
         es.keygroupsave('assistedkills_events', '|assistedkills')
         es.keygroupdelete('assistedkills_events')
      es.loadevents('declare', 'addons/eventscripts/assistedkills/es_assistedkills_events_db.txt')
      es.dbgmsg(0, 'AssistedKills: Unable to load assistedkills.res! Please ensure it is in the ./assistedkills/ directory.')