################################################################################
#  -------------------- PICKUPS ---------------------------------------------- #
################################################################################
# Allows you to create pickable funtional objects anywhere in the map,         #
# such as healthkit and armor, automatically at round-start.                   #
# ---- Requirements: --------------------------------------------------------- #
# * Eventscripts 2.0 or higher                                                 #
# ---------------------------------------------------------------------------- #
# Written by LosNir. copyright 2008 Nir Azuelos. All rights reserved.          #
# Version 1.1 Alpha                                                            #
################################################################################

import psyco
psyco.full()
import es
import playerlib
import keyvalues
import gamethread
import time
import vecmath
import random
import urllib
import cfglib

info = es.AddonInfo()
info.name        = "Pickups"
info.version     = "1.1 Alpha"
info.author      = "Nir Azuelos a.k.a. LosNir"
info.url         = "http://addons.eventscripts.com/addons/view/pickups"
info.basename    = "pickups"
info.description = "Allows you to create pickable funtional objects anywhere in the map, such as healthkit and armor, automatically at round-start."
info.cfgpath     = "cfg/pickups/pickups.cfg"

config = cfglib.AddonCFG(es.ServerVar("eventscripts_gamedir") + "/" + info.cfgpath)

config.text("******************************")
config.text(" PICKUPS CONFIG")
config.text("  by LosNir")
config.text("******************************")
config.text("")

config.text("*********** PICKUPS ***********")
pLog = int(config.cvar("pLog", 1, "Log any error occured into a file?"))
pLogPath = str(config.cvar("pLogPath", "addons/pickups.log", "Path to log file. (Relative to <gamedir>)"))
pCheckForUpdates = int(config.cvar("pCheckForUpdates", 1, "Check for updates onload and map_change?"))
pDBPath = str(config.cvar("pDBPath", "cfg/pickups/pickupsDB.txt", "Path to database file. (Relative to <gamedir>)"))
pCfgPath = str(config.cvar("pCfgPath", "pickups", "Path to config files. (Relative to <gamedir>/cfg) (DO NOT INCLUDE FILENAME, ONLY FOLDER PATH!!!) (RELATIVE TO CFG FOLDER! USE ../ TO GET ONE FOLDER UP)"))
config.text("")
pDefaultRespawnDelayPickup = float(config.cvar("pDefaultRespawnDelayPickup", 25, "Default delay to use for pickup respawn, in seconds, if not already specified in pickupsDB.txt. 0: Disable Respawn"))
pDefaultRespawnDelayMax = float(config.cvar("pDefaultRespawnDelayMax", 2, "Default delay to use for pickup respawn if maxValue reached, in seconds, if not already specified in pickupsDB.txt. 0: Disable Respawn"))
pDefaultRespawnSound = str(config.cvar("pDefaultRespawnSound", "random.choice(('weapons/stunstick/alyx_stunner1.wav', 'weapons/stunstick/alyx_stunner2.wav'))", "Default respawn sound to use for pickup respawn, if not already specified in pickupsDB.txt. Empty: Disable Respawn Sound"))
pDefaultOnReSpawn = str(config.cvar("pDefaultOnReSpawn", "es.server.cmd(''es_fire %i %pn DispatchEffect ManhackSparks'')", "Default respawn command to execute at pickup respawn, if not already specified in pickupsDB.txt. Empty: Disable Respawn Command"))
pDefaultOnSpawn = str(config.cvar("pDefaultOnSpawn", "", "Default spawn command to execute at pickup spawn, if not already specified in pickupsDB.txt. Empty: Disable Spawn Command"))
config.text("")

config.text("*********** SPAWNING MANAGER (RANDOM) ***********")
pSM_minSpawnDistance = float(config.cvar("pSM_minSpawnDistance", 270, "Minimum distance between (CT + T spawning locations) and (pickup) in order to register spot."))
pSM_minPickupDistance = float(config.cvar("pSM_minPickupDistance", 150, "Minimum distance between each random pickup in order to register spot."))
pSM_spawnPointsSamplingTimeMultipiler = float(config.cvar("pSM_spawnPointsSamplingTimeMultipiler", 0.8, "Multipiler for random pickup spawning spots sampling time. Higher value: slower sampling time, Lower Value: faster sampling time"))
pSM_maxSpawnPoints = int(config.cvar("pSM_maxSpawnPoints", 50, "Maximum random pickup spawning spots to sample. If the maximum amount is reached, the oldest spot will be removed in order to keep the spots updated."))
pSM_playerVelocitySample = float(config.cvar("pSM_playerVelocitySample", 0.0, "Player velocity to filter from sampling. (0.0: Sample only moving players) (-1.0: Sample moving and standing players)"))

config.text("")
config.text("- END OF CONFIG -")

config.write()

class CPickups(object):

     # Player
     _playerID = None
     _playerPos = None
     _playerAngle = None

     # Keygroup
     _pickupsDB = None
     _pickupsRecord = None

     # Pickup
     _pIndex = None
     _pPendingValue = {}

     # Spawn points manager
     _pSpawnManager = None

     # Est Banned Entities Value
     _estBannedEntities = None

     # ################ INITALIZE #####################

     def __init__(self):
          # Class initialized, fires BEFORE es_load event! Therefore, eventscripts WILL NOT be available at this time, in theory.
          # CPickupsPrepare initializes any es-independent methods:

          self._initPreCPickups(1)

     def _initLoad(self):
          # Script on-load (event: es_load)
          # ES Is now available, again, in theory. Therefore, intialize the remaining es-dependent methods:

          # Make 'pickups' public
          es.set("Pickups", info.version)
          es.makepublic("Pickups")

          # Execute onload script
          es.server.queuecmd("exec %s/pickups_load.cfg"%pCfgPath)

          # Register server-console methods for within-script use
          if not es.exists("command", "playerPicked"): es.regcmd("playerPicked", "pickups/playerPicked", "The playerPicked method")

          # Register client-say commands for info about pickups
          if not es.exists("command", "!pickups"): es.regsaycmd("!pickups", "pickups/fireAbout", "Prints info about pickups")

          # Remove point_servercommand from EST Banned Entities, kill all of the entities (if required), and then give our own entity
          self._givePointServerCommand()

     def _initUnload(self):
          # Script on-unload (event: es_unload)

          # Restore est_Banned_Entities value
          es.set("est_Banned_Entities", self._estBannedEntities)

          # Cancel delayed SpawnManager._findSpawnPoints
          gamethread.cancelDelayed("SpawnManager._findSpawnPoints")

          # Unregister commands
          es.unregsaycmd("!pickups")

          # Execute onunload script
          es.server.queuecmd("exec %s/pickups_unload.cfg"%pCfgPath)

          # Clear pickups
          for pickupName in self._pickupsRecord["Index"]: es.server.cmd("es_xremove %s"%pickupName)

          # Cancel delayed
          gamethread.cancelDelayed("pickupRespawn")

     def _initPreCPickups(self, updatePickups):
          # Initialize es-independent properties:

          if updatePickups:
               config.execute()
               self._playerID = es.getuserid()
               self._pickupsDB = keyvalues.KeyValues(filename=pDBPath)
               if pCheckForUpdates: self._checkVersion()
          self._pickupsRecord = keyvalues.KeyValues(name="pickupsRecord")
          self._pickupsRecord["Index"] = keyvalues.KeyValues(name="Index")
          self._pickupsRecord["Count"] = keyvalues.KeyValues(name="Count")
          self._pPendingValue = {}
          if pCheckForUpdates: self._checkVersionPrint()
          gamethread.cancelDelayed("pickupRespawn")

     def _initRoundStart(self):
          # Round start (event: es_round_start)

          # Update / Refresh the playerID + pickupsRecord
          # Why: Maybe the 'victim' has left which will cause the script to hang, so we pick a new victim. :->

          if not self._isValidID(): self._initPreCPickups(1)

          # If the server is empty, 0 will be returned which will spam the console, so lets avoid that
          if self._isValidID():

               # Give a point_servercommand to attach an event trigger to any spawned pickup
               self._givePointServerCommand()

               # Initialize basic methods
               self._initPreCPickups(0)

               # Get CT and T spawning points in order to keep minimum pickups distance from pickup spawn
               tSpawn = es.getindexprop(es.getentityindex("info_player_terrorist"), "CBaseEntity.m_vecOrigin")
               ctSpawn = es.getindexprop(es.getentityindex("info_player_counterterrorist"), "CBaseEntity.m_vecOrigin")

               # Initialize spawn manager
               if self._pSpawnManager == None: self._pSpawnManager = SpawnManager()

               # Calculate random pickups spawning spots based on player location
               self._pSpawnManager._findSpawnPoints(tSpawn, ctSpawn)

               # Save current player position/angle and then force the player to look down
               self._savePlayerForce90Deg(self._playerID)

               # The coordinate analyze algorithm which calls the SpawnPickup with every specified coordinate (a real headache)
               for pickup in self._pickupsDB:
                    self._pickupsRecord["Count"][pickup.getName()] = 0
                    pickupFrequency = self._retrieveKeyValue(self._pickupsDB, pickup.getName(), "frequency")
                    for pickupKey in pickup:
                         if "coordinate" in pickupKey.getName():
                              if ("." in pickupKey.getName() and str(es.ServerVar("eventscripts_currentmap")) in pickupKey.getName().split('.')[1]) or "." not in pickupKey.getName():
                                   for coordinate in self._retrieveKeyValue(self._pickupsDB, pickup.getName(), pickupKey.getName()).split(','):
                                        if coordinate == "[RAND]":
                                             for i in range(pickupFrequency):
                                                  pickupCount = self._retrieveKeyValue(self._pickupsRecord, "Count", pickup.getName())
                                                  if pickupCount < pickupFrequency:
                                                       if self._pSpawnManager._readyForSpawning(): self._spawnPickup(pickup.getName(), self._pSpawnManager._getSpawnPoint(), 1)
                                        else: self._spawnPickup(pickup.getName(), coordinate)

               # Set back the player position and angle
               self._restorePlayerForce90Deg(self._playerID)

     def _initPlayerSpawn(self, event_var):
          # Player Spawned (Joined) (event: es_player_spawn)
          # If a player joined into an empty server, the server would be pickup-less!
          # We spawn the pickups when he joins a team, so he will not need to wait for the next round for the pickups to appear:

          if not self._isValidID() and int(event_var["es_userteam"]) > 1: self._initRoundStart()

     # ################ SPAWNING #####################

     def _spawnPickup(self, pickup, coordinate, isTuple = 0, isRespawn = 0, playerID = 0, Force90Deg = 0):
          # This method will spawn a pickup, 
          # Remember to save the 'victim' position/angle before executing this command, and then restoring it afterwards.

          # Create a coordinate tuple if not already
          if not isTuple: coordinate = tuple(coordinate.split())

          # If playerID is not specified we will use the pre-defined 'victim' ID
          if not playerID > 0: playerID = self._playerID

          # Get pickup specific variables
          pickupCount = self._retrieveKeyValue(self._pickupsRecord, "Count", pickup)
          pickupModel = self._retrieveKeyValue(self._pickupsDB, pickup, "model")
          pickupType = self._retrieveKeyValue(self._pickupsDB, pickup, "type")
          pickupOnSpawn = self._retrieveKeyValue(self._pickupsDB, pickup, "onSpawn", 0)
          pickupOnReSpawn = self._retrieveKeyValue(self._pickupsDB, pickup, "onReSpawn", 0)
          if pickupOnSpawn == None and not pDefaultOnSpawn == "": pickupOnSpawn = pDefaultOnSpawn
          if pickupOnReSpawn == None and not pDefaultOnReSpawn == "": pickupOnReSpawn = pDefaultOnReSpawn

          # Set the player position
          es.server.cmd("es_xsetpos %s %s %s %s" % ((playerID, ) + coordinate))
          
          # Force the player to look down, if needed
          if Force90Deg: es.server.cmd("es_xsetang %s 90 0 0" %playerID)

          # Spawn a pickup
          if pickupType == "physics": es.server.cmd("es_xprop_physics_create %s %s" % (playerID, pickupModel))
          elif pickupType == "dynamic": es.server.cmd("es_xprop_dynamic_create %s %s" % (playerID, pickupModel))
          elif pickupType == "entity": es.server.cmd("es_xentcreate %s %s" % (playerID, pickupModel))
          else: self._log("Incorrect entity type for pickup '%s': '%s' (Valid types: physics, dynamic, entity)"%(pickup, pickupType))

          # Give the pickup a new name
          pName = "pickup_%s" % (int(es.ServerVar("eventscripts_lastgive")))
          es.server.cmd("es_xentsetname %s %s" % (playerID, pName))

          # Restore angle
          if Force90Deg: es.server.cmd("es_xsetang %s %s %s %s" % ((playerID, ) + self._playerAngle))

          # Save the pickup index to a temp var
          self._pIndex = int(es.ServerVar("eventscripts_lastgive"))

          # Attach a trigger
          if pickupType == "entity":
               es.server.cmd('es_xfire %s %s addoutput "OnPlayerPickup !activator,color,254 255 255"' % (playerID, pName))
               es.server.cmd('es_xfire %s %s addoutput "OnPlayerPickup point_servercommand,command,es playerPicked %s %s %s"' % (playerID, pName, pickup, self._pIndex, pName))
          else:
               es.server.cmd('es_xfire %s %s addoutput "spawnflags 32"' % (playerID, pName))
               es.server.cmd('es_xfire %s %s addoutput "OnBreak !activator,color,254 255 255"' % (playerID, pName))
               es.server.cmd('es_xfire %s %s addoutput "OnBreak point_servercommand,command,es playerPicked %s %s %s"' % (playerID, pName, pickup, self._pIndex, pName))

          # Save the index for later use
          self._pickupsRecord["Index"][pName] = self._pIndex

          # Parse 'onSpawn' command
          if isRespawn and pickupOnReSpawn: self._parseCmd(pickupOnReSpawn, playerID, self._pIndex, pName, 0)
          elif pickupOnSpawn: self._parseCmd(pickupOnSpawn, playerID, self._pIndex, pName, 0)

          # Increase the pickup count according to name
          self._pickupsRecord["Count"][pickup] += 1

     def _respawnPickup(self, pickup, coordinate, isTuple = 0, isRespawn = 0, pickerID = 0, respawnSound = ""):
          # This will respawn a pickup, it will do all the dirty work of 'victim' manipulation and sound emitting:

          # If playerID is not specified we will use the pre-defined 'victim' ID
          if not pickerID > 0: pickerID = self._playerID

          # Save current player position/angle and then force the player to look down.
          self._savePlayerForce90Deg(pickerID, 0)

          # Spawn the pickup
          self._spawnPickup(pickup, coordinate, isTuple, isRespawn, pickerID, 1)

          # Set back the player position and angle.
          self._restorePlayerForce90Deg(pickerID, 0)
          
          if not respawnSound == "": es.emitsound("entity", self._pIndex, respawnSound, 1, 1)

     # ################ EVENTS #####################

     def _onPickup(self, pickup, pIndex, pName):
          # Event: onPickup
          # Fires at player pickup
          
          # Get the pickerID
          playerList = playerlib.getPlayerList()
          for pid in playerList:
               pickerID = self._searchPicker(pid.userid)
               if not pickerID == -1: break

          pickupType = self._retrieveKeyValue(self._pickupsDB, pickup, "type")

          pickupAction = self._retrieveKeyValue(self._pickupsDB, pickup, "action")

          pickupSound = self._retrieveKeyValue(self._pickupsDB, pickup, "sound", 0)
          pickupRespawnSound = self._retrieveKeyValue(self._pickupsDB, pickup, "respawnSound", 0)

          pickupOnPickup = self._retrieveKeyValue(self._pickupsDB, pickup, "onPickup", 0)

          pickupDone = self._retrieveKeyValue(self._pickupsDB, pickup, "done", 0)
          pickupMaxReached = self._retrieveKeyValue(self._pickupsDB, pickup, "maxReached", 0)

          pickupRespawnDelayMax = self._retrieveKeyValue(self._pickupsDB, pickup, "respawnDelayMax", 0)
          pickupRespawnDelayPickup = self._retrieveKeyValue(self._pickupsDB, pickup, "respawnDelayPickup", 0)

          if pickupRespawnDelayMax == None: pickupRespawnDelayMax = float(pDefaultRespawnDelayMax)
          if pickupRespawnDelayPickup == None: pickupRespawnDelayPickup = float(pDefaultRespawnDelayPickup)

          # Parse respawn sound
          if pickupRespawnSound == None: pickupRespawnSound = pDefaultRespawnSound
          if not pickupRespawnSound == "": pickupRespawnSound = self._parseCmd(pickupRespawnSound, pickerID, pIndex, pName)

          # Process 'action' block
          for Action in pickupAction.split(','):
               Action = Action.split()
               if Action[0] == "playerset":
                    player = playerlib.getPlayer(pickerID)
                    if not self._getPendingValue(pickup, pickerID) > -1: playerValue = player.attributes[Action[1]]
                    else: playerValue = self._getPendingValue(pickup, pickerID)
                    Action[2] = int(eval(Action[2].replace('%', str(playerValue)), {"__builtins__":None}, {"es":es,"random":random}))
                    maxValue = Action[2]
                    for i in range(3, len(Action)):
                         if self._isFloat(Action[i].strip("[]")) > 0: maxValue = float(Action[i].strip("[]"))
                         elif self._isFloat(Action[i].strip("|")) > 0: Delay = float(Action[i].strip("|"))
                    if Action[2] > maxValue and playerValue < maxValue: Action[2] = maxValue
                    if Action[2] <= maxValue:
                         pickupStatus = 1
                         if Delay > 0:
                              self._setPendingValue(pickup, Action[2], pickerID)
                              gamethread.delayed(Delay, self._removePendingValue, (pickup, pickerID))
                              gamethread.delayed(Delay, player.set, (Action[1], Action[2]))
                         else: player.set(Action[1], Action[2])
                         if pickupDone: self._parseCmd(pickupDone, pickerID, pIndex, pName, int(Action[2]-playerValue))
                         if pickupRespawnDelayPickup > 0: gamethread.delayedname(pickupRespawnDelayPickup, "pickupRespawn", self._respawnPickup, (pickup, self._getPlayerPos(pickerID), 1, 1, pickerID, pickupRespawnSound))
                    else:
                         pickupStatus = 0
                         if pickupType == "physics" or pickupType == "dynamic":
                              if pickupRespawnDelayMax > 0: gamethread.delayedname(pickupRespawnDelayMax, "pickupRespawn", self._respawnPickup, (pickup, self._getPlayerPos(pickerID), 1, 1, pickerID, pickupRespawnSound))
                         if pickupMaxReached: self._parseCmd(pickupMaxReached, pickerID, pIndex, pName, maxValue)

          # Parse 'onPickup' command
          if pickupOnPickup: self._parseCmd(pickupOnPickup, pickerID, pIndex, pName, 0)

          # Process 'sound' block
          if pickupStatus and pickupSound:
               for Sound in pickupSound.split(','):
                    Sound = Sound.split()
                    for i in range(2, len(Sound)):
                         if self._isFloat(Sound[i].strip("|")) > 0: Delay = float(Sound[i].strip("|"))
                    if Delay > 0: gamethread.delayed(Delay, es.emitsound, ("player", pickerID, Sound[0], 1, Sound[1]))
                    else: es.emitsound("player", pickerID, Sound[0], 1, Sound[1])

     def _searchPicker(self, id):
          clrRender = es.getplayerprop(id, "CBaseEntity.m_clrRender")
          if clrRender == -2:
               es.server.cmd('es_xfire %s !self color "255 255 255"' % id)
               return id
          return -1

     # ################ MISC #####################

     def _givePointServerCommand(self):
          # Get the banned entities and then remove point_servercommand from the map
          # Don't worry, no security hole here! Every point_servercommand will be killed

          self._estBannedEntities = str(es.ServerVar("est_Banned_Entities"))
          if "point_servercommand" in self._estBannedEntities:
               es.set("est_Banned_Entities", self._estBannedEntities.replace("point_servercommand", ""))
               # Remove point_servercommand from the map if needed
               es.server.cmd("es_xfire %s point_servercommand kill"%self._playerID)

          # Give our own point_servercommand
          es.server.cmd('es_xgive %s "point_servercommand"'%self._playerID)

     def _restorePlayerForce90Deg(self, playerID, Force90Deg = 1):
          # This will set back the player position and angle.

          if Force90Deg: es.server.cmd("es_xsetang %s %s %s %s" % ((playerID, ) + self._playerAngle))
          es.server.cmd("es_xsetpos %s %s %s %s" % ((playerID, ) + self._playerPos))

     def _savePlayerForce90Deg(self, playerID, Force90Deg = 1):
          # This will save the current player position/angle and then force the player to look down.

          self._playerPos =  self._getPlayerPos(playerID)
          self._playerAngle = self._getPlayerAngle(playerID)
          if Force90Deg: es.server.cmd("es_xsetang %s 90 0 0" %playerID)

     def _getPlayerPos(self, playerID):
          return es.getplayerlocation(playerID)

     def _getPlayerAngle(self, playerID):
          return tuple(es.getplayerprop(playerID, "CCSPlayer.m_angEyeAngles[%s]" % x) for x in range(3))

     def _setPendingValue(self, key, value, playerID):
          if not self._getPendingValue(key, playerID) > -1: self._pPendingValue[playerID] = {}
          self._pPendingValue[playerID][key] = value

     def _getPendingValue(self, key, playerID):
          try: return self._pPendingValue[playerID][key]
          except KeyError: return -1

     def _removePendingValue(self, key, playerID):
          del self._pPendingValue[playerID][key]

     def _isFloat(self, n, returnInt=1):
          try: return float(n)
          except ValueError:
               if returnInt: return 0
               else: return str(n)

     def _parseCmd(self, cmd, pickerID=0, pickupID=0, pickupName=0, v=0):
          cmd = cmd.replace('%i', str(pickerID)).replace('%v', str(v)).replace('%pn', str(pickupName)).replace('%p', str(pickupID)).replace("''", '"')
          for Command in cmd.split(';'):
               SeekCommands = Command.split('[')
               for i in range(len(SeekCommands)):
                    if not SeekCommands[i].find(")]") == -1:
                         InnerCommand = SeekCommands[i].split("]")[0]
                         Command = Command.replace('['+InnerCommand+']', str(eval(InnerCommand, {"__builtins__":None}, {"es":es,"random":random})))
               if Command.startswith("es_"): es.server.queuecmd(Command)
               else:
                    try:
                         Results = eval(Command, {"__builtins__":None}, {"es":es,"random":random})
                         if Results == None: return Command
                         else: return Results
                    except: return Command

     def _retrieveKeyValue(self, KeyGroup, Key, keyValue, Fatal = 1):
          # This will retrieve a keyValue

          try: return KeyGroup[Key][keyValue]
          except KeyError:
               if Fatal: self._log("Failed to load keyValue '%s' for pickup '%s'!" %(keyValue, Key))
               return None

     def _isValidID(self):
          if not es.exists("userid", self._playerID): return False
          return True

     def _log(self, msg, logFile = es.ServerVar("eventscripts_gamedir") + "/" + pLogPath):
          es.msg("#multi", "\r\n[#greenPickups#default]#lightgreen Oops! An error has occurred. Please check the log file at")
          es.msg("#lightgreen", "                 '%s' for more details or take a look at your"%pLogPath)
          es.msg("#lightgreen", "                 server's console.\r\n")
          es.log("[Pickups] %s" %msg)
          es.dbgmsg(0, "[Pickups] %s"%msg)
          if pLog:
               fileStream = open(logFile, 'a')
               fileStream.write(time.strftime("%x %X ") + es.ServerVar("eventscripts_currentmap") + " --- "+ msg + "\r\n")
               fileStream.close()

     def _about(self):
          es.msg("#multi", "\r\n[#greenPickups#default]#lightgreen Pickups %s By #greenLosNir#lightgreen. Download:#default %s\r\n"%(info.version, info.url))

     def _checkVersion(self):
          try:
               urlStream = urllib.urlopen("http://addons.eventscripts.com/addons/chklatestver/" + info.basename)
               info.latestVersion = urlStream.read()
               urlStream.close()
          except:
               self._log("Unable to get latest pickups version.")

     def _checkVersionPrint(self):
          if not info.latestVersion == info.version:
               es.msg("#multi", "\n[#greenPickups#default]#lightgreen Pickups %s is oudated! Please update to"%info.version)
               es.msg("#multi", "#lightgreenthe latest version %s by running the server command:#green es_update pickups\r\n"%info.latestVersion)

CPickups = CPickups()

# ##################### SPAWN MANAGER #######################
# ###########################################################

class SpawnManager(object):

     _spawnPoints = []
     _spawnFindJob = 0

     def _findSpawnPoints(self, tSpawn, ctSpawn, forceFind = 0):
          if self._spawnFindJob == 0 or forceFind:
               self._spawnFindJob = 1
               playerNum = 0
               for playerLib in playerlib.getPlayerList("#alive"):
                    playerNum += 1
                    playerID = playerLib.userid
                    m_vecVelocity = tuple(es.getplayerprop(playerID, "CBasePlayer.localdata.m_vecVelocity[%s]"%x) for x in range(3))
                    m_vecVelocity = (m_vecVelocity[0] + m_vecVelocity[1] + m_vecVelocity[2]) / 3
                    if m_vecVelocity > pSM_playerVelocitySample:
                         playerLoc = es.getplayerlocation(playerID)
                         canSpawn = 1
                         tVecDistance = vecmath.distance(tSpawn, playerLoc)
                         ctVecDistance = vecmath.distance(ctSpawn, playerLoc)
                         if tVecDistance >= pSM_minSpawnDistance and ctVecDistance >= pSM_minSpawnDistance:
                              for spawnPoint in self._spawnPoints:
                                   pVecDistance = vecmath.distance(spawnPoint, playerLoc)
                                   if pVecDistance < pSM_minPickupDistance: canSpawn = 0
                         else: canSpawn = 0
                         if canSpawn:
                              if len(self._spawnPoints) > pSM_maxSpawnPoints: del self._spawnPoints[0]
                              self._spawnPoints.append(playerLoc)
               gamethread.delayedname(float(int(playerNum)*pSM_spawnPointsSamplingTimeMultipiler), "SpawnManager._findSpawnPoints", self._findSpawnPoints, (tSpawn, ctSpawn, 1))

     def _getSpawnPoint(self):
          randomSpawnIndex = random.randrange(len(self._spawnPoints))
          randomSpawnLocation = self._spawnPoints[randomSpawnIndex]
          del self._spawnPoints[randomSpawnIndex]
          return randomSpawnLocation

     def _readyForSpawning(self):
          if len(self._spawnPoints) == 0: return False
          return True

# ###################### ES EVNTS ###########################
# ###########################################################

def load():
     CPickups._initLoad()

def unload():
     CPickups._initUnload()

def round_start(event_var):
     CPickups._initRoundStart()

def playerPicked():
     CPickups._onPickup(es.getargv(1), es.getargv(2), es.getargv(3))

def es_map_start(event_var):
     CPickups._initPreCPickups(1)

def player_spawn(event_var):
     CPickups._initPlayerSpawn(event_var)

def fireAbout():
     CPickups._about()