# IPToCountry release 1 by David Bowland
# ./addons/eventscripts/_libs/python/iptocountry.py
# Database provided by http://Software77.net

"""
To use this library, you must first import it:

import iptocountry


The following functions are available:

iptocountry.get_country(ip)
   - Returns full country name, three-letter country abbreviation of IP
   ip = IP address to resolve to country name


iptocountry.get_location_data(ip):
   - Returns a dictionary of location data of IP
   ip = IP address to resolve to location data


iptocountry.update_data(url=None):
   - Syncs server database with internet data--VERY SLOW
   url = URL to download IPToCountry database


iptocountry.get_last_update():
   - Returns the time of the last database update (in seconds since epoch)


The following instance of CountryResolve is publicly exposed:

iptocountry.service


The database for this addon is provided by http://Software77.net. For more information, please see:

http://software77.net/cgi-bin/ip-country/geo-ip.pl

NO CLAIMS ARE MADE AS TO THE ACCURACY OF THIS DATABASE. Please use this library with that in mind.
"""


import es
import os
import sys
import traceback
import urllib
import zipfile


class CountryResolve:
   ip_data = []

   def __init__(self):
      """Loads database for later query"""
      self.base_path = es.getAddonPath('_libs') + '/python/iptocountry/'
      if not os.path.isdir(self.base_path):
         os.mkdir(self.base_path)

      self._load_data()

   def get_location_data(self, ip):
      """
      Returns a dictionary of location data of IP
      Dictionary keys:
         ip_from      = Bottom long IP of location range
         ip_to        = Upper long IP of location range
         registry     = International registry agency through which the IP is registered
         assigned     = Seconds since epoch the IP was assigned
         country_2    = Two-character international country code
         country_3    = Three-character international country code (preferred because more unique)
         country_long = Country name in uppercase
      More information can be found here: http://software77.net/cgi-bin/ip-country/geo-ip.pl
      """
      if not self.ip_data:
         # No database to query
         return {}

      # Formats the IP without port to a list
      if ':' in ip:
         ip = ip[:ip.rfind(':')]
      ip_list = map(int, ip.split('.'))

      # Validates the address for conversion to long format
      if len(ip_list) <> 4:
         raise ValueError, 'Invalid IP address \"%s\"' % ip

      # Converts the IP to long IP format
      long_ip = ip_list[3] + ip_list[2] * 256 + ip_list[1] * 65536 + ip_list[0] * 16777216

      # Finds the IP in the database
      for data in self.ip_data:
         if long_ip <= data['ip_to']:
            return data

      # No data found
      return {}

   def get_country(self, ip):
      """Returns full country name and three-letter country abbreviation of IP"""
      data = self.get_location_data(ip)
      if data:
         return data['country_long'], data['country_3']

      else:
         return 'Unknown', 'Unknown'

   def update_data(self, url=None):
      """
      Syncs server database with internet data--VERY SLOW
      Returns True if update is sucessful, otherwise returns False
      """
      if not url:
         url = 'http://software77.net/cgi-bin/ip-country/geo-ip.pl?action=downloadZ'

      try:
         # Retrieves the new zip file
         zip_path = self.base_path + 'update.zip'
         urllib.urlretrieve(url, zip_path)

         # Extracts the files from the zip
         unzipped = []
         zip = zipfile.ZipFile(zip_path)
         for file in zip.filelist:
            if not os.path.isfile(self.base_path + file.filename):
               zip_output = open(self.base_path + file.filename, 'w')
               zip_output.write(zip.read(file.filename))
               zip_output.close()

               unzipped.append(self.base_path + file.filename)
         zip.close()

         # Parses the database
         if os.path.isfile(self.base_path + 'IpToCountry.csv'):
            self.___parse_database('IpToCountry.csv')
         else:
            self.___parse_database(zip.filelist[0].filename)

         # Removes files that were unzipped
         for file in unzipped:
            os.remove(file)

         # Removes temporary zip file
         os.remove(zip_path)

         # Loads database into memory
         self._load_data()

         # Update sucessful
         return True

      except:
         es.dbgmsg(1, 'IPToCountry update exception:')
         for line in traceback.format_exception(*sys.exc_info()):
            es.dbgmsg(1, line[:-1])

         return False

   def get_last_update(self):
      """Returns the time of the last database update"""
      path = self.base_path + 'iptocountry.db'
      if os.path.isfile(path):
         return os.stat(path).st_mtime
      return 0

   def _load_data(self):
      """
      Internal use recommended: This function is SLOW
      Loads the IP data into memory
      """
      self.ip_data[:] = []
      path = self.base_path + 'iptocountry.db'

      if not os.path.isfile(path):
         es.dbgmsg(0, 'No IPToCountry database found! Downloading database.')
         self.update_data()
         return

      data_db = open(path)

      # Reads the contents of the database to a dictionary
      for line in data_db.readlines():
         data = line.strip().split(',')
         self.ip_data.append({'ip_from':float(data[0]), 'ip_to':float(data[1]), 'registry':data[2], 'assigned':float(data[3]), 'country_2':data[4], 'country_3':data[5], 'country_long':data[6]})

      data_db.close()

   def ___parse_database(self, filename):
      """
      Internal use highly recommended
      Converts raw internet data into plain-text format
      """
      database = []

      path = self.base_path + filename
      if os.path.isfile(path):
         old_db = open(path)
         for line in old_db.readlines():
            if line.startswith('#'): continue

            data = line.strip().replace('\"', '')
            if len(data.split(',')) >= 7:
               database.append(data + '\n')

         old_db.close()

         new_db = open(self.base_path + 'iptocountry.db', 'w')
         # Sorts the database for quicker reference
         new_db.writelines(sorted(database, key=lambda x: float(x.split(',')[0])))
         new_db.close()

      else:
         raise IOError, 'Unable to open raw IP-to-country database.'


# Public instance of CountryResolve
service = CountryResolve()


# Easy-reference functions to the public instance of CountryResolve
def get_country(ip):
   """Returns full country name and three-letter country abbreviation of IP"""
   return service.get_country(ip)
get_country.__doc__ = service.get_country.__doc__


def get_location_data(ip):
   """Returns a dictionary of location data of IP"""
   return service.get_location_data(ip)
get_location_data.__doc__ = service.get_location_data.__doc__


def update_data(url=None):
   """Syncs server database with internet data--VERY SLOW"""
   return service.update_data(url)
update_data.__doc__ = service.update_data.__doc__


def get_last_update():
   """Returns the time of the last database update"""
   return service.get_last_update()
get_last_update.__doc__ = service.get_last_update.__doc__