# SourceRadar release 4 by David Bowland
# ./addons/eventscripts/sourceradar/sourceradar.py

# >>> To configure this addon please see sourceradar.cfg <<<

"""
Plays beeps for players based on the distance of opponents. Requires only ES 2.0+
"""


import es
import gamethread
import os.path

import psyco
psyco.full()


info = es.AddonInfo()
info.name     = 'SourceRadar'
info.version  = '4'
info.url      = 'http://addons.eventscripts.com/addons/view/sourceradar'
info.basename = 'sourceradar'
info.author   = 'SuperDave'


###


class RadarManager:
   enabled  = False
   players  = {}
   sounddir = es.ServerVar('eventscripts_gamedir') + '/sound/'

   def __init__(self):
      self.players = {}


   def enable(self):
      if not self.enabled:
         self.enabled = True

         gamethread.delayedname(es.ServerVar('sourceradar_update_delay'), 'sourceradar_update', self._updateRadar)

   def disable(self):
      if self.enabled:
         self.enabled = False

         gamethread.cancelDelayed('sourceradar_update')
         gamethread.cancelDelayed('sourceradar_beep')

   def checkState(self):
      activate_radar = int(es.ServerVar('sourceradar_activate_radar'))

      if True if not activate_radar else (es.getlivingplayercount(2) <= activate_radar and es.getlivingplayercount(3) <= activate_radar):
         self.enable()

      else:
         self.disable()


   def markDownload(self):
      sound = str(es.ServerVar('sourceradar_beep_sound'))
      if os.path.isfile(self.sounddir + sound):
         es.stringtable('downloadables', 'sound/' + sound)


   def _updateRadar(self):
      teams     = {2:set(), 3:set()}
      locations = {}

      for userid in filter(lambda x: not es.getplayerprop(x, 'CBasePlayer.pl.deadflag'), es.getUseridList()):
         teams[es.getplayerteam(userid)].add(userid)
         locations[userid] = es.getplayerlocation(userid)

      gamethread.cancelDelayed('sourceradar_beep')
      self.players.clear()

      players_on_radar  = int(es.ServerVar('sourceradar_players_on_radar'))
      max_volume_dist   = float(es.ServerVar('sourceradar_max_volume_distance'))
      volume_difference = float(es.ServerVar('sourceradar_radar_distance')) - max_volume_dist
      team_radar        = int(es.ServerVar('sourceradar_team_radar'))

      if team_radar == 1:
         radar_teams = (3,)
      elif team_radar == 2:
         radar_teams = (2,)
      else:
         radar_teams = (2, 3)

      for team in radar_teams:
         for userid in teams[5 - team]:
            loop_location = locations[userid]
            loop_volumes  = sorted(filter(lambda x: x > 0, map(lambda x: (volume_difference - (sum((loop_location[y] - locations[x][y]) ** 2 for y in range(3)) ** 0.5 - max_volume_dist)) / volume_difference, teams[team])), reverse=True)[:players_on_radar]

            if loop_volumes:
               self.players[userid] = loop_volumes

      if self.players:
         self._playBeep()

      gamethread.delayedname(es.ServerVar('sourceradar_update_delay'), 'sourceradar_update', self._updateRadar)

   def _playBeep(self):
      sound = str(es.ServerVar('sourceradar_beep_sound'))

      for userid in self.players.keys():
         loop_player = self.players[userid]

         es.playsound(userid, sound, loop_player[0] if loop_player[0] < 1 else 1.)
         loop_player[:] = loop_player[1:]

         if not loop_player:
            del self.players[userid]

      if self.players:
         gamethread.delayedname(es.ServerVar('sourceradar_beep_delay'), 'sourceradar_beep', self._playBeep)

radar = RadarManager()


###


dict_options = {'sourceradar_update_delay':[2, 'Seconds between radar updates'], 'sourceradar_radar_distance':[1000, 'Distance an opponent must be within to hear a beep'], 'sourceradar_max_volume_distance':[300, 'Distance an opponent must be to hear a beep at the maximum volume'], 'sourceradar_players_on_radar':[5, 'Maximum number of beeps per radar update'], 'sourceradar_beep_delay':[0.3, 'Delay between beeps in seconds'], 'sourceradar_beep_sound':['npc/metropolice/vo/on1.wav', 'Radar beep sound relative to ./sound/--marked for download automatically'], 'sourceradar_team_radar':[0, '0 = both teams have radar, 1 = only Terrorists have radar, 2 = only Counter-Terrorists have radar'], 'sourceradar_activate_radar':[0, 'Radar will be activated when there are this many players or less per team--set to 0 for always-active radar']}

def load():
   for str_option in dict_options:
      es.ServerVar(str_option, dict_options[str_option][0], dict_options[str_option][1])
   if os.path.isfile(es.getAddonPath('sourceradar') + '/sourceradar.cfg'):
      es.server.cmd('es_xmexec ../addons/eventscripts/sourceradar/sourceradar.cfg')
   else:
      es.dbgmsg(0, 'SourceRadar: Unable to load sourceradar.cfg! Please ensure it is in the ./sourceradar/ directory.')

   radar.markDownload()
   radar.checkState()


def es_map_start(event_var):
   radar.disable()
   radar.markDownload()


def round_start(event_var):
   radar.disable()
   radar.checkState()


def player_spawn(event_var):
   radar.checkState()


def player_death(event_var):
   radar.checkState()


def player_disconnect(event_var):
   radar.checkState()


def unload():
   radar.disable()

   for str_option in dict_options:
      es.ServerVar(str_option).set(0)