#pragma semicolon 1

#include <sourcemod>
#include <sdktools>
#include <sdkhooks>

#define CLR_DEFAULT		1
#define CLR_LIGHTGREEN	3
#define CLR_GREEN		4

#define AMP_SHAKE		50.0
#define DUR_SHAKE		1.0

#define MDL_BLUE		"models/stickynades/stickynades_blue.mdl"
#define MDL_RED			"models/stickynades/stickynades_red.mdl"

#define SND_LAUGH		"stickynades/muhahaha.wav"
#define SND_SCREAM		"stickynades/scream.wav"

#define PL_VERSION		"1.2.3"

/* Adapted from DJ Tsunami's Sticky Nades 1.0
*  Stick nade models are from Sticky Nades 1.0
*  http://forums.alliedmods.net/showthread.php?t=102921
*/

/* Changelog:
*  1.1.0 - Cvar snlite_admin_only changed to snlite_admin_only_flags. You can
*  		 - use any string of admin flags you want.
* 		 - Added support for old stickynade models. Use Cvar snlite_hegrenade_model
* 		 - to enable.
*  1.2.0 - Added old Sticky Nades 1.0 sounds. Use convar snlite_emit_sounds to 
* 		 - enable. Normal grenade damage and radius settings now work even if 
* 		 - nades are not sticky.
*  1.2.1 - Fixed. Native "GetEntPropEnt" reported: Property "m_hThrower" not found.
*  1.2.2 - Fixed. Various entity related errors being reported in InitGrenade.
*  1.2.3 - Fixed invalid entities being found.
*/

public Plugin:myinfo =
{
	name        = "StickyNades Lite",
	author      = "TigerOx, Tsunami",
	description = "Allows grenades to stick to walls and players!",
	version     = PL_VERSION,
	url         = ""
}

/**
 * Globals
 */
new Handle:g_hHEGrenadeStickPlayers = INVALID_HANDLE;
new Handle:g_hHEGrenadeStickWalls = INVALID_HANDLE;

new Handle:g_hHEGrenadeNormalPower = INVALID_HANDLE;
new Handle:g_hHEGrenadeNormalRadius = INVALID_HANDLE;
new Handle:g_hHEGrenadeStuckPower = INVALID_HANDLE;
new Handle:g_hHEGrenadeStuckRadius = INVALID_HANDLE;

new Handle:g_hHEGrenadeModel = INVALID_HANDLE;
new Handle:g_hEmitSounds = INVALID_HANDLE;
new Handle:g_hShake = INVALID_HANDLE;

new Handle:g_hNotice = INVALID_HANDLE;
new Handle:g_hAdminFlag = INVALID_HANDLE;

/**
 * Values for speed
 */
new bool:v_HEGrenadeStickPlayers;
new bool:v_HEGrenadeStickWalls;

new Float:v_HEGrenadeNormalPower;
new Float:v_HEGrenadeNormalRadius;
new Float:v_HEGrenadeStuckPower; 
new Float:v_HEGrenadeStuckRadius;

new bool:v_HEGrenadeModel;
new bool:v_EmitSounds;
new bool:v_Shake;

new bool:v_Notice;
new String:v_AdminFlags[64];
new bool:v_CanUseSticky[MAXPLAYERS+1];

public OnPluginStart()
{
	CreateConVar("sm_snlite_version", PL_VERSION, "Allows grenades to stick to walls and players!", FCVAR_DONTRECORD|FCVAR_NOTIFY|FCVAR_PLUGIN|FCVAR_REPLICATED|FCVAR_SPONLY);
	
	g_hHEGrenadeStickPlayers = CreateConVar("snlite_hegrenade_stick_to_players", "1",   "Make HE Grenades stick to players",                    FCVAR_PLUGIN);
	g_hHEGrenadeStickWalls   = CreateConVar("snlite_hegrenade_stick_to_walls",   "1",   "Make HE Grenades stick to walls",                      FCVAR_PLUGIN);
	
	g_hHEGrenadeNormalPower  = CreateConVar("snlite_hegrenade_normal_power",     "100", "Power of a HE grenade when not stuck to a player",     FCVAR_PLUGIN);
	g_hHEGrenadeNormalRadius = CreateConVar("snlite_hegrenade_normal_radius",    "350", "Radius of a HE grenade when not stuck to a player",    FCVAR_PLUGIN);
	g_hHEGrenadeStuckPower   = CreateConVar("snlite_hegrenade_stuck_power",      "300", "Power of a HE grenade when stuck to a player",         FCVAR_PLUGIN);
	g_hHEGrenadeStuckRadius  = CreateConVar("snlite_hegrenade_stuck_radius",     "350", "Radius of a HE grenade when stuck to a player",        FCVAR_PLUGIN);
	
	g_hHEGrenadeModel        = CreateConVar("snlite_hegrenade_model",            "0",   "Use a custom model for HE grenades",                   FCVAR_PLUGIN);
	g_hEmitSounds            = CreateConVar("snlite_emit_sounds",                "0",   "Play sounds when a player gets stuck",                 FCVAR_PLUGIN);
	g_hShake                 = CreateConVar("snlite_shake",                      "1",   "Shake player when hit by a stickynade",                FCVAR_PLUGIN);
	
	g_hNotice                = CreateConVar("snlite_notice",                     "1",   "Display StickyNades information on round start",       FCVAR_PLUGIN);
	g_hAdminFlag             = CreateConVar("snlite_admin_only_flags", 	         "",    "String of admin flags required to use StickyNades", 	FCVAR_PLUGIN);
	
	HookEvent("round_start",    Event_RoundStart);
	
	AutoExecConfig(true, "snlite");
}

public OnConfigsExecuted()
{
	UpdateAllConVars();
}

public ConVarChanged()
{
	UpdateAllConVars();
}

UpdateAllConVars()
{
	v_HEGrenadeStickPlayers = GetConVarBool(g_hHEGrenadeStickPlayers);
	v_HEGrenadeStickWalls = GetConVarBool(g_hHEGrenadeStickWalls);

	v_HEGrenadeNormalPower = GetConVarFloat(g_hHEGrenadeNormalPower);
	v_HEGrenadeNormalRadius = GetConVarFloat(g_hHEGrenadeNormalRadius);
	v_HEGrenadeStuckPower = GetConVarFloat(g_hHEGrenadeStuckPower);
	v_HEGrenadeStuckRadius = GetConVarFloat(g_hHEGrenadeStuckRadius);
	
	v_Shake = GetConVarBool(g_hShake);
	
	v_Notice = GetConVarBool(g_hNotice);
	GetConVarString(g_hAdminFlag, v_AdminFlags, sizeof(v_AdminFlags));
}

public OnMapStart()
{
	//Only update models and sound Convars on mapstart 
	//to ensure files are cached.
	v_HEGrenadeModel = GetConVarBool(g_hHEGrenadeModel);
	v_EmitSounds = GetConVarBool(g_hEmitSounds);
	
	if(v_HEGrenadeModel)
	{
		//Precache models
		PrecacheModel(MDL_BLUE, true);
		PrecacheModel(MDL_RED,  true);
		
		//Add files to downloadtable
		AddFolderToDownloadsTable("materials/models/stickynades");
		AddFolderToDownloadsTable("models/stickynades");
	}
	if(v_EmitSounds)
	{
		//Precache sounds
		PrecacheSound(SND_LAUGH,  true);
		PrecacheSound(SND_SCREAM, true);
		
		//Add files to downloadtable
		AddFolderToDownloadsTable("sound/stickynades");
	}
}

public OnClientPostAdminCheck(client)
{
	//Check if admin flag set
	if(v_AdminFlags[0] != '\0')
	{
		v_CanUseSticky[client] = false;
		
		//Check if player has admin flag	 
		if(GetUserFlagBits(client) & ReadFlagString(v_AdminFlags))
		{
			v_CanUseSticky[client] = true;
		}
	}
	else
	{
		v_CanUseSticky[client] = true;
	}
}

/**
 * SDKHooks
 */
public OnEntityCreated(iEntity, const String:classname[]) 
{
	//Change Grenade model and properties before it is spawned
	if(StrEqual(classname, "hegrenade_projectile")) 
    {	
		HookSingleEntityOutput(iEntity, "OnUser2", InitGrenade, true);
		SetVariantString("OnUser1 !self:FireUser2::0.01:1");
		AcceptEntityInput(iEntity, "AddOutput");
		AcceptEntityInput(iEntity, "FireUser1");
	}
}

public GrenadeTouch(iGrenade, iEntity) 
{
	//Stick if player
	if(v_HEGrenadeStickPlayers && iEntity > 0 && iEntity <= MaxClients)
	{
		StickGrenade(iEntity,iGrenade);
	}
	//Stick to object
	else if(v_HEGrenadeStickWalls && GetEntityMoveType(iGrenade) != MOVETYPE_NONE)
	{
		SetEntityMoveType(iGrenade, MOVETYPE_NONE);
	}
}
/**
 * SDKHooks END
 */

public InitGrenade(const String:output[], iGrenade, activator, Float:delay)
{
	new iClient = GetEntPropEnt(iGrenade, Prop_Send, "m_hOwnerEntity");
	
	if(iClient <= 0 || !v_CanUseSticky[iClient])
		return;
	
	//Set normal grenade properties
	SetEntPropFloat(iGrenade, Prop_Send, "m_flDamage",  v_HEGrenadeNormalPower);
	SetEntPropFloat(iGrenade, Prop_Send, "m_DmgRadius", v_HEGrenadeNormalRadius);
	
	if(v_HEGrenadeModel)
	{
		SetEntityModel(iGrenade, GetEntProp(iGrenade, Prop_Send, "m_iTeamNum") == 2 ? MDL_RED : MDL_BLUE);
		SetEntProp(iGrenade, Prop_Send, "m_clrRender", -1);
	}
	
	//Hook grenade collision for sticky nades
	SDKHook(iGrenade, SDKHook_StartTouch, GrenadeTouch);
}

 StickGrenade(iClient, iGrenade)
{
	//Set to noblock to keep players from getting stuck when nade stops
	SetEntProp(iGrenade, Prop_Send, "m_CollisionGroup",  2);
	
	//stop movement
	SetEntityMoveType(iGrenade, MOVETYPE_NONE);
	
	//set properties
	SetEntPropFloat(iGrenade, Prop_Send, "m_flDamage",  v_HEGrenadeStuckPower);
	SetEntPropFloat(iGrenade, Prop_Send, "m_DmgRadius", v_HEGrenadeStuckRadius);
	
	// Stick grenade to victim
	SetVariantString("!activator");
	AcceptEntityInput(iGrenade, "SetParent", iClient);
	SetVariantString("idle");
	AcceptEntityInput(iGrenade, "SetAnimation");
	
	SetEntProp(iGrenade, Prop_Data, "m_nSolidType", 6);
	SetEntPropVector(iGrenade, Prop_Send, "m_angRotation", Float:{0.0, 45.0, 0.0});
		
	// If shake is enabled, shake victim
	if(v_Shake)
	{
		Shake(iClient, AMP_SHAKE, DUR_SHAKE);
	}
	
	//If sounds are enabled, emit them
	new iThrower = GetEntPropEnt(iGrenade, Prop_Send, "m_hThrower");
	
	if(v_EmitSounds)
	{
		EmitSoundToClient(iClient, SND_SCREAM);
		EmitSoundToClient(iThrower, SND_LAUGH);
	}
	
	//Print stuck message
	PrintToChatAll("%c[StickyNades] %c%N%c stuck %c%N%c with a %cFrag Grenade%c!",
									CLR_GREEN, CLR_LIGHTGREEN, iThrower, CLR_DEFAULT, CLR_LIGHTGREEN, iClient, 
									CLR_DEFAULT, CLR_LIGHTGREEN, CLR_DEFAULT);
}

public Event_RoundStart(Handle:event, const String:name[], bool:dontBroadcast)
{
	if(!v_Notice)
		return;
	
	//Print stickynades announcement
	if(v_HEGrenadeStickPlayers && g_hHEGrenadeStickWalls)
		PrintToChatAll("%c[StickyNades] %cFrag Grenades%c stick to %cwalls%c and %cplayers%c.",
										CLR_GREEN, CLR_LIGHTGREEN, CLR_DEFAULT, CLR_LIGHTGREEN, CLR_DEFAULT, 
										CLR_LIGHTGREEN, CLR_DEFAULT);
	else if(v_HEGrenadeStickPlayers)
		PrintToChatAll("%c[StickyNades] %cFrag Grenades%c stick to %cplayers%c.",
										CLR_GREEN, CLR_LIGHTGREEN, CLR_DEFAULT, CLR_LIGHTGREEN, CLR_DEFAULT);
	else if(g_hHEGrenadeStickWalls)
		PrintToChatAll("%c[StickyNades] %cFrag Grenades%c stick to %cwalls%c.",
										CLR_GREEN, CLR_LIGHTGREEN, CLR_DEFAULT, CLR_LIGHTGREEN, CLR_DEFAULT);
}

 /* Shake
 *  By Tsunami - Sticky Nades 1.0
 * */
stock Shake(iClient, Float:flAmplitude, Float:flDuration)
{
	new Handle:hBf = StartMessageOne("Shake", iClient);
	if(!hBf)
		return;
	
	BfWriteByte(hBf,  0);
	BfWriteFloat(hBf, flAmplitude);
	BfWriteFloat(hBf, 1.0);
	BfWriteFloat(hBf, flDuration);
	EndMessage();
}

 /* AddFolderToDownloadsTable
 *  By Tsunami - Sticky Nades 1.0
 * */
stock AddFolderToDownloadsTable(const String:sDirectory[])
{
	decl String:sFile[64], String:sPath[512];
	new FileType:iType, Handle:hDir = OpenDirectory(sDirectory);
	while(ReadDirEntry(hDir, sFile, sizeof(sFile), iType))     
	{
		if(iType == FileType_File)
		{
			Format(sPath, sizeof(sPath), "%s/%s", sDirectory, sFile);
			AddFileToDownloadsTable(sPath);
		}
	}
}
