# -*- coding:Utf-8 -*-

# ==================================================
#  IMPORTATIONS D'EVENTSCRIPTS
# ==================================================

import es
import langlib

# ==================================================
#  IMPORTATIONS DE PYTHON
# ==================================================

import shelve
import datetime

# ==================================================
#  INFORMATIONS DU SCRIPT
# ==================================================

info = es.AddonInfo()
info.name        = 'Show-Damage'
info.author      = 'L\'In20Cible'
info.version     = '2.0'
info.basename    = 'showdamage'
info.description = 'Permet de voir les dommages causés directement au milieu de son écran.'

# Création d'une variable publique.
es.ServerVar('sd_version', info.version + ' by ' + info.author).makepublic()

# ==================================================
#  VARIABLES GLOBALES
# ==================================================

# Récupération du répertoire de travail.
addonpath = es.getAddonPath('showdamage')

# Création/Chargement d'une base de donnée qui permettra d'entreposer les préférences des joueurs.
settings = shelve.open(addonpath + '/settings.db', writeback = True)

# Chargement du fichier de traduction.
translations = langlib.Strings(addonpath + '/translations.ini')

# Création d'un dictionnaire pour entreposer les dommages causées par les joueurs.
players = {}

# ==================================================
#  VARIABLES DE CONFIGURATION
# ==================================================

# Création des variables globales en fonction des variables de configuration.
sd_status = es.ServerVar('sd_status', 1, 'Status par défaut du Show-Damage pour les nouveaux joueurs.')
sd_advert = es.ServerVar('sd_advert', 1, 'Active/Désactive les annonces du script au début des rounds.')
sd_saycmd = es.ServerVar('sd_saycmd', '!showdamage', 'Définit la commande t\'chat pour activer/désactiver le Show-Damage.')

# ==================================================
#  INITIALISATION DU SCRIPT
# ==================================================

# Fonction éxécutée au chargement du script.
def load():
   # Exécution du fichier de configuration.
   es.server.queuecmd('es_xmexec ../addons/eventscripts/showdamage/configuration.cfg')
   
# Fonction exécutée au chargement d'un nouveau niveau.
def es_map_start(event_var):
   # Exécution du fichier de configuration.
   es.server.queuecmd('es_xmexec ../addons/eventscripts/showdamage/configuration.cfg')
   
   # Création d'une boucle sur chaque joueur contenu dans la base de donnée.
   for player in settings:
      # Récupération de la date lors de la dernière connexion du joueur.
      date = settings[player]['date'].split('-')
      # Récupération de la différence de jours entre sa dernière connexion et la date d'aujourd'hui.
      diff = datetime.date.today() - datetime.date(int(date[0]), int(date[1]), int(date[2]))
      # Vérification qu'il ne s'est pas connecté durant les 7 derniers jours.
      if diff.days >= 7:
         # Suppression du joueur de la base de donnée.
         del settings[player]
   
# Cette fonction est exécuté au déchargement du script.
def unload():
   # Sauvegarde de la base de donnée.
   settings.close()
   
# ==================================================
#  ÉVÉNEMENTS
# ==================================================

# Cette fonction est déclenchée lorsqu'un joueur arrive sur le serveur.
def player_activate(event_var):
   # Récupération de l'identifiant du joueur en numérique.
   userid = int(event_var['userid'])
   
   # Vérification que le joueur qui arrive sur le serveur n'est pas un BOT.
   if es.isbot(event_var['userid']):
      return
      
   # Ajout/Mise à jour des données du joueur dans la base de donnée.
   settings_check(event_var['es_steamid'])

# Cette fonction est exécutée lorsqu'un joueur écrit dans le t'chat.
def player_say(event_var):
   # Récupération de l'identifiant du joueur en numérique.
   userid = int(event_var['userid'])
   
   # Vérification que le joueur n'est pas un BOT.
   if es.isbot(userid):
      return
      
   # Vérification que le texte écrit s'agit bien de la commande pour activer/désactiver le Show-Damage.
   if not str(event_var['text']).lower() == str(sd_saycmd).lower():
      return
      
   # Récupération du steamid du joueur.
   steamid = event_var['es_steamid']
   
   # Ajout/Mise à jour des données du joueur dans la base de donnée.
   settings_check(steamid)
   
   # Vérification que le Show-Damage du joueur est activé.
   if settings[steamid]['status']:
      # Désactivation du status du Show-Damage du joueur.
      settings[steamid]['status'] = 0
      # Envoie du message au joueur pour lui annoncer qu'il vient de désactiver le Show-Damage.
      tell(userid, get_string('disable', get_lang(userid)))
   # Si le Show-Damage du joueur est désactivé.
   else:
      # Activation du Show-Damage du joueur.
      settings[steamid]['status'] = 1
      # Envoie du message au joueur pour lui annoncer qu'il vient d'activer le Show-Damage.
      tell(userid, get_string('enable', get_lang(userid)))
   
# Cette fonction est exécutée lorsqu'un joueur est touché.
def player_hurt(event_var):
   # Récupération de l'identifiant de l'attaquant en numérique.
   attacker = int(event_var['attacker'])
   
   # Vérification que l'attaquant s'agit bien d'un joueur humain.
   if es.isbot(attacker) or not es.exists('userid', attacker):
      return
   
   # Récupération de la quantité des dommages causés en numérique.
   damages = int(event_var['dmg_health'])
   
   # Vérification que l'attaquant n'est pas présent dans le dictionnaire.
   if attacker not in players:
      # Ajout de l'attaquant au dictionnaire.
      players[attacker] = damages
   # Si le joueur est déjà présent dans le dictionnaire.
   else:
      # Ajout des dommages au total des dommages de l'attaquant.
      players[attacker] += damages
      
   # Récupération du "steamid" de l'attaquant.
   attackersteamid = event_var['es_attackersteamid']
      
   # Ajout/Mise à jour des données du joueur dans la base de donnée.
   settings_check(attackersteamid)
      
   # Vérification des préférences de l'attaquant.
   if settings[attackersteamid]['status']:
      # Affichage de la quantité des dommages causés au milieu de l'écran de l'attaquant.
      es.centertell(attacker, '-%i HP' % damages)
      
# Cette fonction est exécuté lorsqu'un joueur meurt.
def player_death(event_var):
   # Récupération de l'identifiant du joueur en numérique.
   userid = int(event_var['userid'])
   
   # Affichage du total des dommages causés sur le round en cours.
   showdamage(userid)
   
   # Vérification que le joueur est dans le dictionnaire.
   if userid in players:
      # Retrait du joueur du dictionnaire.
      del players[userid]
   
# Cette fonction est exécutée lorsqu'un round se termine.
def round_end(event_var):
   # Vérification qu'il y a bien des joueurs dans le dictionnaire.
   if not len(players):
      return
      
   # Création d'une boucle sur chacun des joueurs présent dans le dictionnaire.
   for player in players:
      # Affichage du total des dommages causés sur le round en cours.
      showdamage(int(player))
         
   # Réinitialisation du dictionnaire.
   players.clear()
   
# Cette fonction est exécutée lorsqu'un nouveau round commence.
def round_start(event_var):
   # Vérification que les annonces sont bien activées.
   if not int(sd_advert):
      return
      
   # Envoie du message à tout les joueurs pour leur indiquer la commande qu'ils doivent utiliser pour activer/désactiver le Show-Damage.
   message('advert', {'saycmd' : str(sd_saycmd).lower()})
   
# ==================================================
#  UTILITAIRES
# ==================================================

# Cette fonction est exécutée lors de la vérification de la présence d'un joueur dans la base de donnée.
def settings_check(steamid):
   # Vérification que le joueur n'est pas déjà présent dans la base de donnée.
   if steamid not in settings:
      # Ajout du joueur à la base de donnée.
      settings[steamid] = {'date'   : datetime.date.today(), 
                           'status' : 1 if int(sd_status) else 0}
   # Si le joueur est déjà présent dans la base de donnée.
   else:
      # Mise à jour de la date lors de sa dernière connexion.
      settings[steamid]['date'] = datetime.date.today()
      
# Cette fonction est exécutée pour afficher la quantité de dommages qu'a causé un joueur sur le round en cours.
def showdamage(userid):
   # Vérification que le joueur est bien présent dans le dictionnaire.
   if userid not in players:
      # Envoie du message pour lui annoncer qu'il n'a pas causé de dommage sur le round en cours.
      tell(userid, get_string('nodamage', get_lang(userid)))
      return
      
   # Récupération du "steamid" du joueur.
   steamid = es.getplayersteamid(userid)
   
   # Ajout/Mise à jour des données du joueur dans la base de donnée.
   settings_check(steamid)
   
   # Vérification des préférences du joueur.
   if settings[steamid]['status']:
      # Récupération de la quantité des dommages causés par le joueur.
      damages = players[userid]
      
      # Vérification que le joueur à bien causé des dommages.
      if damages > 0:
         # Envoie du message pour lui annoncer la quantité de dommage qu'il a causé sur le round en cours.
         tell(userid, get_string('showdamage', get_lang(userid), {'damages' : damages}))
      else:
         # Envoie du message pour lui annoncer qu'il n'a pas causé de dommage sur le round en cours.
         tell(userid, get_string('nodamage', get_lang(userid)))
   
# Cette fonction est utilisée pour récupérer le language d'un joueur.
def get_lang(userid):
   # Vérification que le joueur s'agit bien d'un humain.
   if es.isbot(userid):
      return
      
   # Retourne le language du joueur.
   return langlib.getLangAbbreviation(es.getclientvar(userid, 'cl_language'))
   
# Cette fonction est utilisée pour récupérer la traduction d'un message.
def get_string(string, language, tokens=None):
   # Retourne la traduction d'un message.
   return translations(string, tokens, language)
   
# Cette fonction est utilisée pour envoyer un message à tout les joueurs.
def message(string, tokens=None):
   # Récupération de tout les joueurs humain présent sur le serveur.
   playerlist = [userid for userid in es.getUseridList() if not es.isbot(userid)]
   
   # S'il y a bien des joueurs humain présent sur le serveur.
   if not len(playerlist):
      return
      
   # Création d'une boucle sur chaque joueur humain présent sur le serveur.
   for player in playerlist:
      # Envoie du message dans la langue du joueur.
      tell(player, get_string(string, get_lang(player), tokens))
      
# Cette fonction est utilisé pour envoyer un message à un seul joueur.
def tell(userid, string):
   es.tell(userid, '#multi', '#green[Show-Damage] %s' % string)