# ExtendedEvents release 4b by David Bowland
# ./addons/eventscripts/extendedevents/extendedevents.py

# >>> To configure this addon please see extendedevents.cfg (created whent he addon is first loaded) <<<

"""
Allows easy creation of custom events. Requires only ES 2.0.248c+
"""


import cfglib
import es
import os.path
import sys

import psyco
psyco.full()


info = es.AddonInfo()
info.name     = 'ExtendedEvents'
info.version  = '4'
info.basename = 'extendedevents'
info.url      = 'http://addons.eventscripts.com/addons/view/' + info.basename
info.author   = 'SuperDave'


###


config = cfglib.AddonCFG(es.getAddonPath(info.basename) + '/' + info.basename + '.cfg')

config.text(info.name + ' release %s options' % info.version)
config.text('./addons/eventscripts/%(basename)s/%(basename)s.cfg' % {'basename':info.basename})
config.text(info.url)
config.text('')
config.text('Load this addon with: es_load ' + info.basename)
config.text('\n')

config.text('***** Tick option *****')
ticks_to_update = config.cvar('extendedevents_tickstoupdate', 16, 'Number of ticks between ExtendedEvents updates')
config.text('Lower numbers mean higher event accuracy but more CPU usage--1 minimum.')
config.text('\n')

config.text('***** Event options *****')
config.text('')
config.text('Use the following server command to load individual extended events:')
config.text('')
config.text('extendedevents_load <"name">')
config.text('\n')

config.text('Place your extendedevents_load commands below:')
config.command('extendedevents_load')
config.text('\n')

config.text('Uncomment the following extendedevents_load commands as desired:')
config.text('\n')

config.text(' extendedevents_load addon_action')
config.text('')
config.text('addon_action')
config.text('- Fires when an addon is loaded, unloaded, enabled, or disabled')
config.text('NOTE: addon_action does not fire for addons that are reloaded (es_reload)')
config.text('   event_var(name)            - Name of the addon performing the action')
config.text('   event_var(action)          - Action performed ("LOAD", "UNLOAD", "ENABLE", "DISABLE")')
config.text('   event_var(type)            - Type of addon performing the action ("TXT", "PY")')
config.text('\n')

config.text('extendedevents_load player_capturearea // DoDS only')
config.text('')
config.text('player_capturearea')
config.text('- Fires when a player enters or exits a capture area')
config.text('Special thanks to Ojii for this event!')
config.text('   event_var(userid)          - Userid of the player who entered a capture area')
config.text('   event_var(action)          - Action performed ("ENTER", "EXIT")')
config.text('   event_var(entityindex)     - Entity index of capture area entered (0 if area exited)')
config.text('')
config.text('dod_flag_captured')
config.text('- Fires when a flag is captured')
config.text('Special thanks to EmbouT for this event!')
config.text('   event_var(cp)              - Valve-assigned control point index')
config.text('   event_var(cpname)          - Control point name')
config.text('   event_var(cappers)         - String of players who captured the flag separated with a comma')
config.text('   event_var(bomb)            - 1 if bomb exploded else 0')
config.text('\n')

config.text('extendedevents_load player_dod // DoDS only')
config.text('')
config.text('player_mgdeploy')
config.text('- Fires when a player deploys or retracts his or her MG')
config.text('   event_var(userid)          - Userid of the player who deployed or retracted the MG')
config.text('   event_var(state)           - State of MG ("DEPLOY", "RETRACT")')
config.text('')
config.text('player_prone')
config.text('- Fires when a player goes prone or stands from being prone')
config.text('   event_var(userid)          - Userid of the player who change prone status')
config.text('   event_var(state)           - Prone status ("ENTER", "EXIT")')
config.text('')
config.text('player_sprint')
config.text('- Fires when a player starts or stops sprinting')
config.text('   event_var(userid)          - Userid of the player who started or stopped sprinting')
config.text('   event_var(state)           - State of player sprint ("START", "STOP")')
config.text('')
config.text('player_stamina')
config.text('- Fires when a player\'s stamina changes')
config.text('   event_var(userid)          - Userid of the player whose stamina changed')
config.text('   event_var(newvalue)        - New stamina value')
config.text('   event_var(oldvalue)        - Old stamina value')
config.text('   event_var(change)          - Change in stamina value')
config.text('')
config.text('player_stun')
config.text('- Fires when a player\'s stunned alpha or duration changes')
config.text('   event_var(userid)          - Userid of player whose stunned alpha or duration changed.')
config.text('   event_var(duration)        - Duration of stun')
config.text('   event_var(alpha)           - Alpha value of stun')
config.text('')
config.text('player_unstun')
config.text('- Fires when a player\'s stunned duration expires or stunned alpha or duration is reset to 0')
config.text('   event_var(userid)          - Userid of the player who is no longer stunned')
config.text('\n')

config.text('extendedevents_load player_drop')
config.text('')
config.text('player_activeweapon')
config.text('- Fires when a player changes active (equipped) weapons')
config.text('   event_var(userid)          - Userid of the player who changed weapons')
config.text('   event_var(weapon)          - Weapon the player changed to')
config.text('   event_var(old_weapon)      - Weapon the player changed from')
config.text('')
config.text('player_drop')
config.text('- Fires when a player uses the drop command to drop a weapon')
config.text('   event_var(userid)          - Userid of the player who used the drop command')
config.text('   event_var(weapon)          - Weapon the player dropped')
config.text('   event_var(index)           - Index of the weapon the player dropped')
config.text('\n')

config.text('extendedevents_load player_duck')
config.text('')
config.text('player_duck')
config.text('- Fires when a player ducks')
config.text('   event_var(userid)          - Userid of the player who ducked')
config.text('')
config.text('player_unduck')
config.text('- Fires when a player stands up after ducking')
config.text('   event_var(userid)          - Userid of the player who stopped ducking')
config.text('\n')

config.text('extendedevents_load player_flag')
config.text('')
config.text('map_end')
config.text('- Fires when the map ends')
config.text('NONE: This event only fires when at least one player is connected to the server')
config.text('No event variables accompany this event.')
config.text('')
config.text('player_air')
config.text('- Fires when a player is in the air (from jumping, falling, etc.)')
config.text('   event_var(userid)          - Userid of the player who is in the air')
config.text('')
config.text('player_flag')
config.text('- Fires when a player\'s flag changes')
config.text('   event_var(userid)          - Userid of the player who changed flags')
config.text('   event_var(flag)            - Flag number (0 to 31--see below)')
config.text('   event_var(state)           - State of the flag ("ON", "OFF")')
config.text('')
config.text('player_land')
config.text('- Fires when a player lands on the ground after being in the air (jumping, falling, etc.)')
config.text('   event_var(userid)          - Userid of the player who landed on the ground')
config.text('\n')

config.text('extendedevents_load player_flashed // CSS only')
config.text('')
config.text('player_flashed')
config.text('- Fires when a player\'s flash alpha or duration changes')
config.text('   event_var(userid)          - Userid of the player who was blinded')
config.text('   event_var(attacker)        - Userid of the player who flashbanged the blind player (if applicable)')
config.text('   event_var(alpha)           - Alpha intensity of flash')
config.text('   event_var(duration)        - Duration of flash')
config.text('')
config.text('player_unflashed')
config.text('- Fires when a player\'s flash delay expires or the flash alpha or duration is set to 0')
config.text('NOTE: This assumes host_timescale is 1')
config.text('   event_var(userid)          - Userid of the player who is no longer blinded')
config.text('\n')

config.text('extendedevents_load player_health')
config.text('')
config.text('player_health')
config.text('- Fires when a player\'s health changes')
config.text('   event_var(userid)          - Userid of the player who changed health')
config.text('   event_var(old_health)      - Old amount of health')
config.text('   event_var(change_health)   - Amount of health the player received (negative for health taken)')
config.text('\n')

config.text('extendedevents_load player_jump // DoDS only')
config.text('')
config.text('player_jump')
config.text('Thanks to EmbouT for submitting this event!')
config.text('- Fires when a player jumps')
config.text('   event_var(userid)          - Userid of the player who jumped')
config.text('\n')

config.text('extendedevents_load player_money // CSS only')
config.text('')
config.text('player_money')
config.text('- Fires when a player\'s money changes')
config.text('   event_var(userid)          - Userid of the player who changed money')
config.text('   event_var(amount)          - Amount of money the player now has')
config.text('   event_var(old_amount)      - Old amount of money ')
config.text('   event_var(change_amount)   - Amount of money the player received (negative for money taken)')
config.text('')
config.text('weapon_purchase')
config.text('- Fires when a player purchases a weapon')
config.text('   event_var(userid)          - Userid of the player who purchased a weapon')
config.text('   event_var(weapon)          - Weapon purchased (omitting "weapon_" prefix)')
config.text('   event_var(cost)            - Cost of weapon')
config.text('   event_var(index)           - Index of weapon')
config.text('\n')

config.text('extendedevents_load player_move')
config.text('')
config.text('player_move')
config.text('- Fires when a player moves on the x-, y-, or z-axis')
config.text('   event_var(userid)          - Userid of the player who moved')
config.text('   event_var(x)               - Distance the player moved on the x-axis')
config.text('   event_var(y)               - Distance the player moved on the y-axis')
config.text('   event_var(z)               - Distance the player moved on the z-axis')
config.text('\n')

config.text('extendedevents_load player_nightvision // CSS only')
config.text('')
config.text('player_nightvision')
config.text('- Fires when a player toggles his or her night vision goggles')
config.text('   event_var(userid)          - Userid of the player who toggled his or her night vision')
config.text('   event_var(state)           - State of the player\'s night vision ("ON", "OFF")')
config.text('\n')

config.text('extendedevents_load player_ping // CSS, DoDS, HL2DM, TF2 only')
config.text('')
config.text('player_ping')
config.text('- Fires when a player\'s ping changes')
config.text('   event_var(userid)          - Userid of the player whose water state changed')
config.text('   event_var(ping)            - New ping of player')
config.text('\n')

config.text('extendedevents_load player_water')
config.text('')
config.text('player_water')
config.text('- Fires when a player\'s water state changes')
config.text('   event_var(userid)          - Userid of the player whose water state changed')
config.text('   event_var(state)           - 0 = not in the water, 1 = water below the waist, 2 = water above waist and head above water, 3 = under water')
config.text('\n')

config.text('extendedevents_load player_zone // CSS only')
config.text('')
config.text('player_bombzone')
config.text('- Fires when a player enters or exits a bomb zone')
config.text('   event_var(userid)          - Userid of the player who entered or exited a bomb zone')
config.text('   event_var(action)          - Action performed ("ENTER", "EXIT")')
config.text('   event_var(site)            - Bombsite entered or exited ("A", "B", "UNKNOWN")')
config.text('')
config.text('player_buyzone')
config.text('- Fires when a player enters or exits a buy zone')
config.text('   event_var(userid)          - Userid of the player who entered or exited a buy zone')
config.text('   event_var(action)          - Action performed ("ENTER", "EXIT")')
config.text('')
config.text('player_rescuezone')
config.text('- Fires when a player enters or exits a hostage rescue zone')
config.text('   event_var(userid)          - Userid of the player who entered or exited a rescue zone')
config.text('   event_var(action)          - Action performed ("ENTER", "EXIT")')
config.text('\n')

config.text('If you want to create custom events an example event can be found at:')
config.text('./addons/eventscripts/extendedevents/events/example_event.py')
config.text('\n')

config.text('Each event also provides event_var(es_steamid), event_var(es_username), and all the other event_vars that accompany')
config.text('event_var(userid). More information can be found here: http://www.eventscripts.com/pages/Extended_event_variables')
config.text('\n')

config.text('Individual extended events can be unloaded with the following server command:')
config.text('')
config.text('extendedevents_unload <"name">')
config.text('')
config.text('Due to the fact events can be added and removed above, this command will not be used by most users')
config.text('\n')

config.text('***** Tick usage information *****')
config.text('')
config.text('Most ExtendedEvents events work by monitoring certain information such as player or entity properties. That information is')
config.text('updated every few gameframes ("ticks") as dictated by the extendedevents_tickstoupdate variable. ExtendedEvents attempts to')
config.text('ease the burden updating places on the server by staggering event updates.')
config.text('')
config.text('A full output of tick usage can be obtained with the following server command:')
config.text('')
config.text('extendedevents_tickusage [count=1000]')
config.text('')
config.text('NOTE: This command requires players to be connected to the server for accurate usage information. However, while usage information')
config.text('is being compiled the server will be unresponsive likely resulting in connected players being dropped. When possible it is')
config.text('recommended to use this command when only bots are connected to the server.')
config.text('')
config.text('Usage information is compiled by timing the execution of event updates. The "count" parameter (optional) dictates how many times')
config.text('each event will be updated for timing purposes. Higher count numbers mean more accuracy but more time required to determine')
config.text('tick usage. The default value is 1000.')
config.text('')
config.text('When reading the tick usage output higher numbers mean more server burden.')
config.text('\n')

config.text('***** Flag values *****')
config.text('')
config.text('Below are the flag numbers and the corresponding flags for player_flag:')
config.text('NOTE: Some flags do not apply to players')
config.text('')
config.text(' 0 -- FL_ONGROUND')
config.text(' 1 -- FL_DUCKING')
config.text(' 2 -- FL_WATERJUMP')
config.text(' 3 -- FL_ONTRAIN')
config.text(' 4 -- FL_INRAIN')
config.text(' 5 -- FL_FROZEN')
config.text(' 6 -- FL_ATCONTROLS')
config.text(' 7 -- FL_CLIENT')
config.text(' 8 -- FL_FAKECLIENT')
config.text(' 9 -- FL_INWATER')
config.text('10 -- FL_FLY')
config.text('11 -- FL_SWIM')
config.text('12 -- FL_CONVEYOR')
config.text('13 -- FL_NPC')
config.text('14 -- FL_GODMODE')
config.text('15 -- FL_NOTARGET')
config.text('16 -- FL_AIMTARGET')
config.text('17 -- FL_PARTIALGROUND')
config.text('18 -- FL_STATICPROP')
config.text('19 -- FL_GRAPHED')
config.text('20 -- FL_GRENADE')
config.text('21 -- FL_STEPMOVEMENT')
config.text('22 -- FL_DONTTOUCH')
config.text('23 -- FL_BASEVELOCITY')
config.text('24 -- FL_WORLDBRUSH')
config.text('25 -- FL_OBJECT')
config.text('26 -- FL_KILLME')
config.text('27 -- FL_ONFIRE')
config.text('28 -- FL_DISSOLVING')
config.text('29 -- FL_TRANSRAGDOLL')
config.text('30 -- FL_UNBLOCKABLE_BY_PLAYER')

config.write()

es_debug = es.ServerVar('eventscripts_debug')


###


class Event(object):
   def __init__(self, gamename):
      pass

   def mapStart(self):
      pass

   def playerSpawn(self, userid):
      pass

   def playerDisconnect(self, userid):
      pass

   def tick(self, playerlist):
      pass

   def clientFilter(self, userid, args):
      pass

   def unload(self):
      pass

   def fire(self, name, event_var={}):
      if int(es_debug) > 0: # Faster than just calling es.dbgmsg
         es.dbgmsg(1, 'ExtendedEvents: Firing event %s' % name)

      es.event('initialize', name)
      try:
         for var_name, (var_type, var_value) in event_var.items():
            es.event('set' + var_type, name, var_name, var_value)

      finally:
         es.event('fire', name)

   def loadRes(self, name):
      es.loadevents('declare', 'addons/eventscripts/extendedevents/events/%s.res' % name)

   def registerForEvent(self, name, event, callback):
      es.addons.registerForEvent(__import__('extendedevents.events.%s' % name).events.__dict__[name], event, callback)

   def unregisterForEvent(self, name, event):
      es.addons.unregisterForEvent(__import__('extendedevents.events.%s' % name).events.__dict__[name], event)


###


class EventManager(object):
   gamename   = str(es.ServerVar('eventscripts_gamedir')).replace('\\', '/').rsplit('/', 1)[~0]
   tick_count = 0

   def __init__(self):
      self.events    = {}
      self.tickorder = [[]]

      es.addons.registerClientCommandFilter(self._cc_filter)
      es.addons.registerTickListener(self._ticker)


   def registerEvent(self, name, callback):
      if name not in self.events:
         event = self.events[name] = callback(self.gamename)
         try:
            event.mapStart()
         finally:
            self.sortTickOrder()

   def unregisterEvent(self, name):
      if name in self.events:
         try:
            self.events[name].unload()
         finally:
            del self.events[name]

         modulename = 'extendedevents.events.%s' % name
         if modulename in sys.modules:
            del sys.modules[modulename]

         self.sortTickOrder()

   def sortTickOrder(self):
      if self.events:
         tick_total = int(ticks_to_update)
         eventlist  = self.events.items()
         step       = float(tick_total) / len(eventlist)

         self.tickorder = [[] for x in range(tick_total)]
         for x, event in enumerate(eventlist):
            pos = int(round(x * step, 2))
            self.tickorder[pos if pos < tick_total else (tick_total - 1)].append(event)

      else:
         self.tickorder = [[]]

   def importEvent(self, name):
      try:
         return reload(__import__('extendedevents.events.%s' % name).events.__dict__[name]).getEvent()

      except ImportError:
         return None


   def mapStart(self):
      for event in self.events:
         try:
            self.events[event].mapStart()
         finally:
            pass

   def playerSpawn(self, userid):
      for event in self.events:
         try:
            self.events[event].playerSpawn(userid)
         finally:
            pass

   def playerDisconnect(self, userid):
      for event in self.events:
         try:
            self.events[event].playerDisconnect(userid)
         finally:
            pass

   def _ticker(self):
      self.tick_count = (self.tick_count + 1) if self.tick_count < len(self.tickorder) - 1 else 0

      tick_events = self.tickorder[self.tick_count]
      if tick_events:
         useridlist = es.getUseridList()
         for name, event in tick_events:
            try:
               event.tick(useridlist)

            except:
               es.excepter(*sys.exc_info()) # Let ES handle the exception normally
               self.unregisterEvent(name)

   def _cc_filter(self, userid, args):
      for event in self.events:
         try:
            self.events[event].clientFilter(userid, args)
         finally:
            pass

      return True

   def _unload(self):
      es.addons.unregisterClientCommandFilter(self._cc_filter)
      es.addons.unregisterTickListener(self._ticker)

      for event in self.events.keys():
         self.unregisterEvent(event)


   def tickUsage(self, count=1000):
      """
      Use this function when bots are connected to see tick usage information for loaded events.
      Bots must be used as connected clients will be disconnected but players are required to test usage.
      """
      import timeit

      es.dbgmsg(0, 'ExtendedEvents tick usage:\n')

      if self.events:
         longestname = max(map(len, self.events))
         total_time  = 0
         for x, tick in enumerate(self.tickorder):
            if not tick: continue

            tick_time = 0
            es.dbgmsg(0, ' Tick %s:' % x)

            for name, event in sorted(tick):
               event_time = timeit.Timer('event.tick(useridlist)', "useridlist = __import__('es').getUseridList();event = __import__('extendedevents').extendedevents.events.events['%s']" % name).timeit(count)
               tick_time += event_time
               es.dbgmsg(0, '  %s - %s' % (name.center(longestname), event_time))

            total_time += tick_time
            es.dbgmsg(0, ' Tick time: %s\n' % tick_time)

         es.dbgmsg(0, 'Total ticks: %s' % len(self.tickorder))
         es.dbgmsg(0, 'Total time: %s' % total_time)
         es.dbgmsg(0, 'Average tick time: %s' % (total_time / len(self.tickorder)))

      else:
         es.dbgmsg(0, ' No events loaded')

events = EventManager()


def loadEvent(name):
   event = events.importEvent(name)
   if not event: raise ImportError, 'Error importing event \'%s\'' % name

   events.registerEvent(name, event)


def unloadEvent(name):
   events.unregisterEvent(name)


###


def load_cmd():
   """
   extendedevents_load <"name">
   """
   if es.getargc() == 2:
      loadEvent(es.getargv(1))

   else:
      es.dbgmsg(0, 'Syntax: extendedevents_load <\"name\">')

if not es.exists('command', 'extendedevents_load'):
   es.regcmd('extendedevents_load', 'extendedevents/load_cmd', 'extendedevents_load <\"name\">\nLoads an ExtendedEvent')


def unload_cmd():
   """
   extendedevents_unload <"name">
   """
   if es.getargc() == 2:
      unloadEvent(es.getargv(1))

   else:
      es.dbgmsg(0, 'Syntax: extendedevents_unload <\"name\">')

if not es.exists('command', 'extendedevents_unload'):
   es.regcmd('extendedevents_unload', 'extendedevents/unload_cmd', 'extendedevents_unload <\"name\">\nUnloads an ExtendedEvent')


def tickusage_cmd():
   """
   extendedevents_tickusage [count=1000]
   """
   argc = es.getargc()
   if argc in (1, 2):
      events.tickUsage(1000 if argc == 1 else int(es.getargv(1)))

   else:
      es.dbgmsg(0, 'Syntax: extendedevents_tickusage [count=1000]')

if not es.exists('command', 'extendedevents_tickusage'):
   es.regcmd('extendedevents_tickusage', 'extendedevents/tickusage_cmd', 'extendedevents_tickusage [count=1000]\nDisplays ExtendedEvents tick usage information\nWARNING: Do not use while players are connected to server')


###


def load():
   config.execute()


def es_map_start(event_var):
   events.mapStart()


def player_spawn(event_var):
   events.playerSpawn(int(event_var['userid']))


def player_disconnect(event_var):
   events.playerDisconnect(int(event_var['userid']))


def unload():
   events._unload()